package com.schbrain.archetype.initializer.util;

import cn.hutool.extra.spring.SpringUtil;
import com.schbrain.archetype.initializer.param.ArchetypeGenerateParam;
import com.schbrain.common.exception.BaseException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.URIish;
import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.ProtectedBranchesApi;
import org.gitlab4j.api.models.Group;
import org.gitlab4j.api.models.Project;
import org.gitlab4j.api.models.ProtectedBranch;

import java.net.URISyntaxException;
import java.nio.file.Path;
import java.util.List;

/**
 * @author liaozan
 * @since 2023/7/26
 */
@Slf4j
public class GitlabUtils {

    private static final List<String> ADDITIONAL_BRANCHES = List.of("release", "main");

    public static void initGitRepo(Path directory, Long repoGroupId, ArchetypeGenerateParam param) {
        try {
            Git git = Git.init().setInitialBranch("develop").setDirectory(directory.toFile()).call();
            git.add().addFilepattern(".").call();
            git.commit().setAllowEmpty(true).setAuthor("initializer", "no-reply@schbrain.com").setMessage("Initial Commit").call();
            for (String branch : ADDITIONAL_BRANCHES) {
                git.checkout().setName(branch).setCreateBranch(true).call();
            }

            if (repoGroupId != null) {
                GitLabApi gitLabApi = SpringUtil.getBean(GitLabApi.class);
                Group group = gitLabApi.getGroupApi().getGroup(repoGroupId);
                String groupPath = group.getFullPath();
                String projectName = param.getArtifactId();
                String repositoryUrl = String.format("%s/%s/%s.git", gitLabApi.getGitLabServerUrl(), groupPath, projectName);
                URIish urIish = new URIish(repositoryUrl);
                git.remoteSetUrl().setRemoteName("origin").setRemoteUri(urIish).call();
                git.push()
                        .setRemote("origin")
                        .setCredentialsProvider(SpringUtil.getBean(CredentialsProvider.class))
                        .setPushAll()
                        .setForce(true)
                        .call();
                Project project = getProject(gitLabApi, groupPath, projectName);
                if (project == null) {
                    return;
                }
                updateProtectedBranches(gitLabApi, project);
                updateProjectDefaultBranch(gitLabApi, project);
            }
        } catch (GitAPIException | GitLabApiException | URISyntaxException e) {
            log.warn("Git repo init failed", e);
            throw new BaseException("Git仓库初始化失败");
        }
    }

    private static Project getProject(GitLabApi gitLabApi, String groupPath, String projectName) throws GitLabApiException {
        String projectPath = String.format("%s/%s", groupPath, projectName);
        return gitLabApi.getProjectApi().getProject(projectPath);
    }

    private static void updateProjectDefaultBranch(GitLabApi gitLabApi, Project project) throws GitLabApiException {
        project.setDefaultBranch("main");
        gitLabApi.getProjectApi().updateProject(project);
    }

    private static void updateProtectedBranches(GitLabApi gitlabApi, Project project) throws GitLabApiException {
        ProtectedBranchesApi protectedBranchesApi = gitlabApi.getProtectedBranchesApi();
        String projectPath = project.getPathWithNamespace();
        List<ProtectedBranch> protectedBranches = protectedBranchesApi.getProtectedBranches(projectPath);
        if (CollectionUtils.isNotEmpty(protectedBranches)) {
            for (ProtectedBranch protectedBranch : protectedBranches) {
                protectedBranchesApi.unprotectBranch(projectPath, protectedBranch.getName());
            }
        }
    }

}
