package com.schbrain.archetype.initializer.runner;

import cn.hutool.core.io.FileUtil;
import cn.hutool.system.SystemUtil;
import com.schbrain.archetype.initializer.config.properties.ArchetypeProperties;
import com.schbrain.archetype.initializer.util.MavenUtils;
import com.schbrain.common.exception.BaseException;
import com.schbrain.common.util.EnvUtils;
import com.schbrain.framework.support.spring.OnceApplicationContextEventListener;
import lombok.extern.slf4j.Slf4j;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.transport.TagOpt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.stereotype.Component;

import java.io.File;

/**
 * @author liaozan
 * @since 2022/3/20
 */
@Slf4j
@Component
@EnableConfigurationProperties(ArchetypeProperties.class)
public class ArchetypePreparer extends OnceApplicationContextEventListener<ContextRefreshedEvent> {

    @Autowired
    private ArchetypeProperties archetypeProperties;

    @Override
    protected void onEvent(ContextRefreshedEvent event) {
        if (!EnvUtils.runningOnCloudPlatform(getApplicationContext().getEnvironment())) {
            return;
        }
        installArchetype();
    }

    private void installArchetype() {
        File archetypeTemplateDirectory = createArchetypeDirectory();
        checkoutArchetype(archetypeTemplateDirectory);
        MavenUtils.install(archetypeTemplateDirectory.getAbsolutePath());
    }

    private File createArchetypeDirectory() {
        String tempDir = SystemUtil.getUserInfo().getTempDir();
        File archetypeTemplateDirectory = new File(tempDir, "archetype");
        FileUtil.del(archetypeTemplateDirectory);
        return archetypeTemplateDirectory;
    }

    private void checkoutArchetype(File archetypeTemplateDirectory) {
        try (Git ignored = Git.cloneRepository()
                .setTagOption(TagOpt.FETCH_TAGS)
                .setNoCheckout(false)
                .setDirectory(archetypeTemplateDirectory)
                .setURI(archetypeProperties.getSourceUrl())
                .call()) {
            log.info("Success checkout archetype source from {} to {}", archetypeProperties.getSourceUrl(), archetypeTemplateDirectory);
        } catch (GitAPIException e) {
            throw new BaseException("Failed to checkout archetype source from git repository", e);
        }
    }

}
