package com.schbrain.archetype.initializer.service;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ZipUtil;
import com.schbrain.archetype.initializer.maven.MavenUtils;
import com.schbrain.archetype.initializer.param.ArchetypeGenerateParam;
import com.schbrain.archetype.initializer.response.PreviewFileTree;
import com.schbrain.archetype.initializer.runner.ArchetypePreparer;
import com.schbrain.common.util.ServletUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.util.StreamUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * @author liaozan
 * @since 2022/3/20
 */
@Service
public class ArchetypeService {

    private final Map<String, String> archetypeNameCache = new ConcurrentHashMap<>();

    @Autowired
    private ArchetypePreparer archetypePreparer;

    public String generate(ArchetypeGenerateParam param) {
        String generateId = MavenUtils.generate(param);
        archetypeNameCache.put(generateId, param.getArtifactId());
        return generateId;
    }

    public List<PreviewFileTree> preview(String id) throws FileNotFoundException {
        File generatedFiles = getGeneratedFiles(id);
        return List.of(buildFileTree(generatedFiles));
    }

    public void download(String id) throws IOException {
        File generatedFiles = ZipUtil.zip(getGeneratedFiles(id));
        FileInputStream inputStream = new FileInputStream(generatedFiles);
        HttpServletResponse response = ServletUtils.getResponse();
        response.addHeader(HttpHeaders.CONTENT_DISPOSITION, contentDisposition(generatedFiles));
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        StreamUtils.copy(inputStream, response.getOutputStream());
    }

    public boolean update() {
        archetypePreparer.installArchetype();
        return true;
    }

    private PreviewFileTree buildFileTree(File root) {
        File[] fileItems = root.listFiles();
        if (ArrayUtil.isEmpty(fileItems)) {
            return new PreviewFileTree(root);
        }
        List<PreviewFileTree> children = Arrays.stream(fileItems)
                .map(fileItem -> {
                    PreviewFileTree childrenFileTree;
                    if (fileItem.isDirectory()) {
                        childrenFileTree = buildFileTree(fileItem);
                    } else {
                        childrenFileTree = new PreviewFileTree(fileItem);
                    }
                    return childrenFileTree;
                })
                .sorted(Comparator.comparing(PreviewFileTree::getIsFile))
                .collect(Collectors.toList());
        return new PreviewFileTree(root, children);
    }

    private File getGeneratedFiles(String id) throws FileNotFoundException {
        File archetypeDirectory = MavenUtils.getArchetypeDirectory(id);
        if (!archetypeDirectory.exists()) {
            throw new FileNotFoundException(archetypeDirectory.getAbsolutePath());
        }
        String artifactId = archetypeNameCache.get(id);
        return new File(archetypeDirectory, artifactId);
    }

    private String contentDisposition(File file) {
        return ContentDisposition
                .attachment()
                .filename(file.getName())
                .build()
                .toString();
    }

}