package com.schbrain.archetype.initializer.service;

import cn.hutool.cache.impl.TimedCache;
import cn.hutool.core.io.file.PathUtil;
import cn.hutool.core.util.ZipUtil;
import cn.hutool.system.SystemUtil;
import com.schbrain.archetype.initializer.maven.MavenUtils;
import com.schbrain.archetype.initializer.param.ArchetypeGenerateParam;
import com.schbrain.archetype.initializer.response.PreviewFileTree;
import com.schbrain.common.exception.BaseException;
import com.schbrain.common.web.utils.ServletUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author liaozan
 * @since 2022/3/20
 */
@Slf4j
@Service
public class ArchetypeService {

    private static final String GITKEEP = ".gitkeep";

    private final TimedCache<String, String> archetypeNameCache = new TimedCache<>(Duration.ofHours(1).toMillis());

    private final Path gitKeepFile = createGitKeepFile(SystemUtil.getUserInfo().getTempDir());

    public String generate(ArchetypeGenerateParam param) throws FileNotFoundException {
        String generateId = MavenUtils.generate(param);
        archetypeNameCache.put(generateId, param.getArtifactId());
        preview(generateId);
        return generateId;
    }

    public List<PreviewFileTree> preview(String id) throws FileNotFoundException {
        File generatedFiles = getGeneratedFiles(id);
        return List.of(buildFileTree(generatedFiles));
    }

    public void download(String id) throws IOException {
        File generatedFiles = ZipUtil.zip(getGeneratedFiles(id));
        FileInputStream inputStream = new FileInputStream(generatedFiles);
        HttpServletResponse response = ServletUtils.getResponse();
        response.addHeader(HttpHeaders.CONTENT_DISPOSITION, contentDisposition(generatedFiles));
        response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
        StreamUtils.copy(inputStream, response.getOutputStream());
    }

    private PreviewFileTree buildFileTree(File root) {
        File[] fileItems = Optional.ofNullable(root.listFiles()).orElse(new File[0]);
        if (fileItems.length == 0) {
            Path target = Path.of(root.getPath(), GITKEEP);
            fileItems = new File[]{PathUtil.copy(gitKeepFile, target).toFile()};
        }
        List<PreviewFileTree> children = Arrays.stream(fileItems)
                .map(fileItem -> {
                    PreviewFileTree childrenFileTree;
                    if (fileItem.isDirectory()) {
                        childrenFileTree = buildFileTree(fileItem);
                    } else {
                        childrenFileTree = new PreviewFileTree(fileItem);
                    }
                    return childrenFileTree;
                })
                .sorted(Comparator.comparing(PreviewFileTree::getIsFile))
                .collect(Collectors.toList());
        return new PreviewFileTree(root, children);
    }

    private Path createGitKeepFile(String directory) {
        try {
            Path created = Files.createFile(Path.of(directory, GITKEEP));
            Files.writeString(created, "This file is used to keep an empty folder when git uploads, If you put other files in the folder, delete this file please");
            return created;
        } catch (IOException e) {
            throw new BaseException("Failed to create .gitkeep file", e);
        }
    }

    private File getGeneratedFiles(String id) throws FileNotFoundException {
        File archetypeDirectory = MavenUtils.getArchetypeDirectory(id);
        if (!archetypeDirectory.exists()) {
            throw new FileNotFoundException(archetypeDirectory.getAbsolutePath());
        }
        String artifactId = archetypeNameCache.get(id);
        if (!StringUtils.hasText(artifactId)) {
            throw new FileNotFoundException("文件已过期,请重新生成");
        }
        return new File(archetypeDirectory, artifactId);
    }

    private String contentDisposition(File file) {
        return ContentDisposition
                .attachment()
                .filename(file.getName())
                .build()
                .toString();
    }

}
