package com.schbrain.archetype.initializer.maven;

import cn.hutool.core.io.FileUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.schbrain.archetype.initializer.param.ArchetypeGenerateParam;
import com.schbrain.common.exception.BaseException;
import com.schbrain.common.util.IdWorker;
import com.schbrain.common.util.JacksonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.maven.cli.MavenCli;
import org.apache.maven.settings.Mirror;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.io.DefaultSettingsWriter;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.URIish;
import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.ProtectedBranchesApi;
import org.gitlab4j.api.models.Group;
import org.gitlab4j.api.models.Project;
import org.gitlab4j.api.models.ProtectedBranch;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;

import static org.apache.maven.cli.MavenCli.MULTIMODULE_PROJECT_DIRECTORY;

/**
 * @author liaozan
 * @since 2022/3/20
 */
@Slf4j
public class MavenUtils {

    private static final File SETTINGS_FILE = new File(MavenCli.USER_MAVEN_CONFIGURATION_HOME, "settings.xml");

    private static final List<String> ADDITIONAL_BRANCHES = List.of("release", "main");

    static {
        System.getProperties().setProperty(MULTIMODULE_PROJECT_DIRECTORY, "$M2_HOME");
        initSettingsFile();
    }

    public static void install(String workDirectory) {
        log.info("Prepare to install archive from : {}", workDirectory);
        MavenCli mavenCli = new MavenCli();
        mavenCli.doMain(getInstallArgs(SETTINGS_FILE.getAbsolutePath()), workDirectory, System.out, System.err);
        log.info("Success install archive from : {}", workDirectory);
    }

    public static String generate(ArchetypeGenerateParam param) {
        log.info("Prepare to generate archetype project: {}", JacksonUtils.toJsonString(param, true));
        String id = IdWorker.getIdStr();
        File archetype = getArchetypeDirectory(id);
        String outputDirectory = archetype.getAbsolutePath();
        String[] args = getArchetypeGenerateArgs(param, SETTINGS_FILE.getAbsolutePath(), outputDirectory);
        MavenCli.doMain(args, null);
        log.info("Generate archetype project at {}", outputDirectory);
        if (param.isInitGitRepo()) {
            Path directory = Paths.get(archetype.getAbsolutePath(), param.getArtifactId());
            initGitRepo(directory, param.getRepoGroupId(), param);
        }
        return id;
    }

    public static File getArchetypeDirectory(String id) {
        String tmpDirPath = FileUtil.getTmpDirPath();
        return new File(tmpDirPath, id);
    }

    private static void initGitRepo(Path directory, Long repoGroupId, ArchetypeGenerateParam param) {
        try {
            Git git = Git.init().setInitialBranch("develop").setDirectory(directory.toFile()).call();
            git.add().addFilepattern(".").call();
            git.commit().setAllowEmpty(true).setAuthor("initializer", "no-reply@schbrain.com").setMessage("Initial Commit").call();
            for (String branch : ADDITIONAL_BRANCHES) {
                git.checkout().setName(branch).setCreateBranch(true).call();
            }

            if (repoGroupId != null) {
                GitLabApi gitLabApi = SpringUtil.getBean(GitLabApi.class);
                Group group = gitLabApi.getGroupApi().getGroup(repoGroupId);
                String groupPath = group.getFullPath();
                String projectName = param.getArtifactId();
                String repositoryUrl = String.format("%s/%s/%s.git", gitLabApi.getGitLabServerUrl(), groupPath, projectName);
                URIish urIish = new URIish(repositoryUrl);
                git.remoteSetUrl().setRemoteName("origin").setRemoteUri(urIish).call();
                git.push()
                        .setRemote("origin")
                        .setCredentialsProvider(SpringUtil.getBean(CredentialsProvider.class))
                        .setPushAll()
                        .setForce(true)
                        .call();
                Project project = getProject(gitLabApi, groupPath, projectName);
                if (project == null) {
                    return;
                }
                updateProjectDefaultBranch(gitLabApi, project);
                updateProtectedBranches(gitLabApi, project);
            }
        } catch (GitAPIException | GitLabApiException | URISyntaxException e) {
            log.warn("Git repo init failed", e);
            throw new BaseException("Git仓库初始化失败");
        }
    }

    private static void updateProjectDefaultBranch(GitLabApi gitLabApi, Project project) throws GitLabApiException {
        project.setDefaultBranch("main");
        gitLabApi.getProjectApi().updateProject(project);
    }

    private static Project getProject(GitLabApi gitLabApi, String groupPath, String projectName) throws GitLabApiException {
        String projectPath = String.format("%s/%s", groupPath, projectName);
        return gitLabApi.getProjectApi().getProject(projectPath);
    }

    private static void initSettingsFile() {
        Settings settings = new Settings();
        settings.setLocalRepository("/data/maven/repository");

        Mirror mirror = new Mirror();
        mirror.setId("aliyun");
        mirror.setUrl("https://maven.aliyun.com/repository/public");
        mirror.setMirrorOf("central");
        settings.addMirror(mirror);

        Server snapshotServer = new Server();
        snapshotServer.setId("libs-snapshots");
        snapshotServer.setUsername("developer");
        snapshotServer.setPassword("good-Style2018");

        Server releaseServer = new Server();
        releaseServer.setId("libs-releases");
        releaseServer.setUsername("developer");
        releaseServer.setPassword("good-Style2018");

        settings.addServer(snapshotServer);
        settings.addServer(releaseServer);

        DefaultSettingsWriter settingsWriter = new DefaultSettingsWriter();
        try {
            settingsWriter.write(SETTINGS_FILE, Collections.emptyMap(), settings);
        } catch (IOException e) {
            log.warn("Failed to write settings file", e);
        }
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static String[] getInstallArgs(String settingsFile) {
        return new String[]{
                "-B",
                "clean",
                "install",
                "-Dmaven.test.skip=true",
                String.format("-gs=%s", settingsFile)
        };
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static String[] getArchetypeGenerateArgs(ArchetypeGenerateParam param, String settingsFile, String outputDirectory) {
        return new String[]{
                "-B",
                "archetype:generate",
                "-Dmaven.test.skip=true",
                "-DarchetypeGroupId=com.schbrain.kp",
                "-DarchetypeArtifactId=schbrain-archetype",
                "-DarchetypeVersion=1.0-SNAPSHOT",
                String.format("-gs=%s", settingsFile),
                String.format("-DoutputDirectory=%s", outputDirectory),
                String.format("-DgroupId=%s", param.getGroupId()),
                String.format("-DartifactId=%s", param.getArtifactId()),
                String.format("-Dversion=%s", param.getVersion()),
                String.format("-Dpackage=%s", param.getPackageName()),
                String.format("-DsubModuleNamePrefix=%s", param.getSubModuleNamePrefix()),
        };
    }

    private static void updateProtectedBranches(GitLabApi gitlabApi, Project project) throws GitLabApiException {
        ProtectedBranchesApi protectedBranchesApi = gitlabApi.getProtectedBranchesApi();
        String projectPath = project.getPathWithNamespace();
        List<ProtectedBranch> protectedBranches = protectedBranchesApi.getProtectedBranches(projectPath);
        if (CollectionUtils.isNotEmpty(protectedBranches)) {
            for (ProtectedBranch protectedBranch : protectedBranches) {
                protectedBranchesApi.unprotectBranch(projectPath, protectedBranch.getName());
            }
        }
    }

}
