package com.schbrain.archetype.initializer.util;

import cn.hutool.core.io.FileUtil;
import com.schbrain.archetype.initializer.param.ArchetypeGenerateParam;
import com.schbrain.common.util.IdWorker;
import com.schbrain.common.util.JacksonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.cli.MavenCli;
import org.apache.maven.settings.Mirror;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.io.DefaultSettingsWriter;

import java.io.File;
import java.io.IOException;
import java.util.Collections;

import static org.apache.maven.cli.MavenCli.MULTIMODULE_PROJECT_DIRECTORY;

/**
 * @author liaozan
 * @since 2022/3/20
 */
@Slf4j
public class MavenUtils {

    private static final File SETTINGS_FILE = new File(MavenCli.USER_MAVEN_CONFIGURATION_HOME, "settings.xml");

    static {
        System.getProperties().setProperty(MULTIMODULE_PROJECT_DIRECTORY, "$M2_HOME");
        initSettingsFile();
    }

    public static void install(String workDirectory) {
        log.info("Prepare to install archive from : {}", workDirectory);
        MavenCli mavenCli = new MavenCli();
        mavenCli.doMain(getInstallArgs(SETTINGS_FILE.getAbsolutePath()), workDirectory, System.out, System.err);
        log.info("Success install archive from : {}", workDirectory);
    }

    public static Pair<String, File> generate(ArchetypeGenerateParam param) {
        log.info("Prepare to generate archetype project: {}", JacksonUtils.toJsonString(param, true));
        String id = IdWorker.getIdStr();
        File archetype = getArchetypeDirectory(id);
        String outputDirectory = archetype.getAbsolutePath();
        String[] args = getArchetypeGenerateArgs(param, SETTINGS_FILE.getAbsolutePath(), outputDirectory);
        MavenCli.doMain(args, null);
        log.info("Generate archetype project at {}", outputDirectory);
        return ImmutablePair.of(id, archetype);
    }

    public static File getArchetypeDirectory(String id) {
        String tmpDirPath = FileUtil.getTmpDirPath();
        return new File(tmpDirPath, id);
    }

    private static void initSettingsFile() {
        Settings settings = new Settings();
        settings.setLocalRepository("/data/maven/repository");

        Mirror mirror = new Mirror();
        mirror.setId("aliyun");
        mirror.setUrl("https://maven.aliyun.com/repository/public");
        mirror.setMirrorOf("central");
        settings.addMirror(mirror);

        Server snapshotServer = new Server();
        snapshotServer.setId("libs-snapshots");
        snapshotServer.setUsername("developer");
        snapshotServer.setPassword("good-Style2018");

        Server releaseServer = new Server();
        releaseServer.setId("libs-releases");
        releaseServer.setUsername("developer");
        releaseServer.setPassword("good-Style2018");

        settings.addServer(snapshotServer);
        settings.addServer(releaseServer);

        DefaultSettingsWriter settingsWriter = new DefaultSettingsWriter();
        try {
            settingsWriter.write(SETTINGS_FILE, Collections.emptyMap(), settings);
        } catch (IOException e) {
            log.warn("Failed to write settings file", e);
        }
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static String[] getInstallArgs(String settingsFile) {
        return new String[]{
                "-B",
                "clean",
                "install",
                "-Dmaven.test.skip=true",
                String.format("-gs=%s", settingsFile)
        };
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static String[] getArchetypeGenerateArgs(ArchetypeGenerateParam param, String settingsFile, String outputDirectory) {
        return new String[]{
                "-B",
                "archetype:generate",
                "-Dmaven.test.skip=true",
                "-DarchetypeGroupId=com.schbrain.kp",
                "-DarchetypeArtifactId=schbrain-archetype",
                "-DarchetypeVersion=1.0-SNAPSHOT",
                String.format("-gs=%s", settingsFile),
                String.format("-DoutputDirectory=%s", outputDirectory),
                String.format("-DgroupId=%s", param.getGroupId()),
                String.format("-DartifactId=%s", param.getArtifactId()),
                String.format("-Dversion=%s", param.getVersion()),
                String.format("-Dpackage=%s", param.getPackageName()),
                String.format("-DsubModuleNamePrefix=%s", param.getSubModuleNamePrefix()),
        };
    }

}
